/*
 *	LCD interface
 *	Uses routines from delay.c
 *	This code will interface to a standard LCD controller
 *	like the Hitachi HD44780. It uses it in 4 bit mode, with
 *	the hardware connected as follows (the standard 14 pin 
 *	LCD connector is used):
 *	
 *	PORTD bits 4-7 are connected to the LCD data bits 4-7 (high nibble)
 *	PORTD bit  1    is connected to the LCD RS input (register select)
 *	PORTD bit  3    is connected to the LCD EN bit (enable)
 *	PORTD bit  2    is connected to the LCD RW bit 
 *	
 *	To use these routines, set up the port I/O (TRISA, TRISB) then
 *	call lcd_init(), then other routines as required.
 *	
 */

#include	"lcd.h"

 
static bit LCD_RS	@ ((unsigned)&PORTD*8+1);	// Register select
static bit LCD_EN	@ ((unsigned)&PORTD*8+3);	// Enable
static bit LCD_RW	@ ((unsigned)&PORTD*8+2);	// Read Write

#define	LCD_STROBE	((LCD_EN = 1),(LCD_EN=0))


/* write a byte to the LCD in 4 bit mode */

void lcd_write(unsigned char c)
{
 PORTD = (PORTD & 0x0F) |  (c & 0xF0);
 LCD_STROBE;
 PORTD = (PORTD & 0x0F) |  (c << 4);
 LCD_STROBE;
 DelayUs(40);
}

/*
 * 	Clear and home the LCD
 */

void
clrscr(void)
{
 LCD_RW = 0;
 LCD_RS = 0;
 lcd_write(0x1);
 DelayMs(2);
}

/* write a string of chars to the LCD */


void
lcd_puts(const char * s)
{
 DDRD=DDRD & 0b00000001;
 LCD_RW = 0;
 LCD_RS = 1;	// write characters
 while(*s) lcd_write(*s++);
}


/* write one character to the LCD */
/*
void putch(char c)
{
 LCD_RW = 0;
 LCD_RS = 1;	// write characters 

 PORTD = (PORTD & 0x0F) |  (c & 0xF0);
 LCD_STROBE;
 PORTD = (PORTD & 0x0F) |  (c << 4);
 LCD_STROBE;
 DelayUs(40);


 if (c== 0x0A) 			//   \n case	
 {
  while (!TRMT1);
  TXREG1 = 13;
  while (!TRMT1);
  TXREG1 = 10;
 }
 else
 {
  while (!TRMT1); 
  TXREG1 = c;
 }
}
*/

/*
 * Go to the specified position
 */

void lcd_goto(unsigned char pos)
{
 LCD_RW = 0;
 LCD_RS = 0;
 lcd_write(0x80+pos);
}

void gotoxy(unsigned char x, unsigned char y)
{
 LCD_RW = 0;
 LCD_RS = 0;
 switch(y)
 {
  case 1 : lcd_write(x+0x7F);	break;
  case 2 : lcd_write(x+0xBF);	break;
  case 3 : lcd_write(x+0x8F);	break;
  case 4 : lcd_write(x+0xCF);	break;
 }
}

void cursor_on()
{
 LCD_RS = 0;
 lcd_write(0b00001111);
}
	
void cursor_off()
{
 LCD_RS = 0;
 lcd_write(0b00001100);
}

/* initialise the LCD - put into 4 bit mode */

void init_LCD(void)
{
 DDRD=DDRD & 0b00000001;
 LCD_RW = 0;
 LCD_RS = 0;	// write control bytes
 DelayMs(15);	// power on delay
 PORTD = 0x30;	// attention!
 LCD_STROBE;
 DelayMs(5);
 LCD_STROBE;
 DelayUs(100);
 LCD_STROBE;
 DelayMs(5);
 PORTD = 0x20;	// set 4 bit mode
 LCD_STROBE;
 DelayUs(40);
 lcd_write(0x28);	// 4 bit mode, 1/16 duty, 5x8 font
 lcd_write(0x08);	// display off
 lcd_write(0x0F);	// display on, blink curson on
 lcd_write(0x06);	// entry mode
 contrast(contrast_value_global);
 cursor_off();
}
